﻿/**
 * server.js Ã¢â‚¬â€ Socket.IO + MySQL (mysql2) Live Chat
 * Features: DM, Groups, Voice recording indicator, Instant read ticks
 * Node 18+
 */
"use strict";

const http   = require("http");
const express = require("express");
const { Server } = require("socket.io");
const mysql  = require("mysql2/promise");
const crypto = require("crypto");

const PORT = Number(process.env.PORT || 3000);
const TOKEN_SECRET = process.env.CHAT_TOKEN_SECRET || "CHANGE_THIS_TO_A_LONG_RANDOM_SECRET_64CHARS_MIN";
const DB_HOST = process.env.DB_HOST || "localhost";
const DB_NAME = process.env.DB_NAME || "nximon_db";
const DB_USER = process.env.DB_USER || "nximon_user";
const DB_PASS = process.env.DB_PASS || "userpassword";
const ALLOWED_ORIGINS = (process.env.ALLOWED_ORIGINS || "https://chat.nximon.online,https://www.chat.nximon.online,https://node.chat.nximon.online")
  .split(",").map(s => s.trim()).filter(Boolean);

function base64urlDecode(str) {
  str = str.replace(/-/g, "+").replace(/_/g, "/");
  while (str.length % 4) str += "=";
  return Buffer.from(str, "base64");
}
function verifyToken(token) {
  try {
    const parts = String(token || "").split(".");
    if (parts.length !== 2) return null;
    const [b64, sigb] = parts;
    const sig  = base64urlDecode(sigb);
    const calc = crypto.createHmac("sha256", TOKEN_SECRET).update(b64).digest();
    if (sig.length !== calc.length || !crypto.timingSafeEqual(sig, calc)) return null;
    const p = JSON.parse(base64urlDecode(b64).toString("utf8"));
    if (!p || !p.uid || !p.exp) return null;
    if (Number(p.exp) < Math.floor(Date.now() / 1000)) return null;
    return { uid: Number(p.uid) };
  } catch { return null; }
}

function dmRoom(a, b) {
  return "dm_" + Math.min(a, b) + "_" + Math.max(a, b);
}

async function main() {
  const app = express();
  app.set("trust proxy", 1);
  app.get("/", (req, res) => res.send("OK"));
  app.get("/health", (req, res) => res.json({ ok: true, ts: Date.now() }));

  const server = http.createServer(app);

  const io = new Server(server, {
    cors: {
      origin: (origin, cb) => {
        if (!origin) return cb(null, true);
        if (ALLOWED_ORIGINS.includes(origin)) return cb(null, true);
        return cb(new Error("CORS blocked"), false);
      },
      methods: ["GET", "POST"],
      credentials: true
    },
    transports: ["websocket", "polling"]
  });

  const pool = await mysql.createPool({
    host: DB_HOST, user: DB_USER, password: DB_PASS, database: DB_NAME,
    waitForConnections: true, connectionLimit: 20, charset: "utf8mb4"
  });

  try {
    await pool.execute("SELECT 1");
    console.log("OK DB connected");
  } catch(e) {
    console.error("DB connection failed:", e.message);
  }

  const userSockets = new Map(); // uid -> Set(socketId)
  const socketUser  = new Map(); // socketId -> uid
  const recordingMap = new Map(); // uid -> { roomId, startedAt }

  function setOnline(uid, socketId) {
    if (!userSockets.has(uid)) userSockets.set(uid, new Set());
    userSockets.get(uid).add(socketId);
  }
  function setOffline(uid, socketId) {
    const set = userSockets.get(uid);
    if (!set) return true;
    set.delete(socketId);
    if (set.size === 0) { userSockets.delete(uid); return true; }
    return false;
  }
  function onlineIds() { return Array.from(userSockets.keys()); }

  function emitToUser(targetUid, event, data) {
    const sockets = userSockets.get(Number(targetUid));
    if (!sockets) return;
    sockets.forEach(sid => {
      const s = io.sockets.sockets.get(sid);
      if (s) s.emit(event, data);
    });
  }

  io.use((socket, next) => {
    const token = socket.handshake.auth?.token || socket.handshake.query?.token || "";
    const p = verifyToken(token);
    if (!p) return next(new Error("unauthorized"));
    socket.userId = p.uid;
    next();
  });

  io.on("connection", async (socket) => {
    const uid = socket.userId;
    socketUser.set(socket.id, uid);
    setOnline(uid, socket.id);
    console.log("User " + uid + " connected (" + socket.id + ")");

    try {
      const [rows] = await pool.execute(
        "SELECT DISTINCT IF(from_id=?, to_id, from_id) AS other_id FROM messages WHERE from_id=? OR to_id=?",
        [uid, uid, uid]
      );
      for (const row of rows) {
        socket.join(dmRoom(uid, Number(row.other_id)));
      }
      try {
        const [convRows] = await pool.execute(
          "SELECT conversation_id FROM conversation_members WHERE user_id=?", [uid]
        );
        convRows.forEach(r => socket.join("c:" + r.conversation_id));
      } catch(e) {}
    } catch(e) { console.error("join rooms error:", e.message); }

    socket.emit("presence:init", { onlineUserIds: onlineIds() });
    io.emit("presence:update", { userId: uid, online: true });

    /* TYPING */
    socket.on("typing", ({ conversationId, typing }) => {
      if (!conversationId) return;
      const room = String(conversationId);
      socket.to(room).emit("typing", { conversationId: room, userId: uid, typing: !!typing });
      const dmMatch = room.match(/^dm_(\d+)_(\d+)$/);
      if (dmMatch) {
        const [, a, b] = dmMatch;
        const otherUid = Number(a) === uid ? Number(b) : Number(a);
        emitToUser(otherUid, "typing", { conversationId: room, userId: uid, typing: !!typing });
      }
    });

    /* VOICE RECORDING INDICATOR */
    socket.on("voice:recording", ({ conversationId, recording }) => {
      if (!conversationId) return;
      const room = String(conversationId);
      const isRecording = !!recording;

      if (isRecording) {
        recordingMap.set(uid, { roomId: room, startedAt: Date.now() });
      } else {
        recordingMap.delete(uid);
      }

      const payload = { userId: uid, recording: isRecording, conversationId: room };

      // Broadcast to room (for group or if both already in room)
      socket.to(room).emit("voice:recording", payload);

      // Also directly emit to the other user's sockets (DM)
      const dmMatch = room.match(/^dm_(\d+)_(\d+)$/);
      if (dmMatch) {
        const [, a, b] = dmMatch;
        const otherUid = Number(a) === uid ? Number(b) : Number(a);
        emitToUser(otherUid, "voice:recording", payload);
      }
    });

    /* MESSAGE SEND */
    socket.on("message:send", async (payload) => {
      try {
        const convId = String(payload?.conversationId || "");
        const providedMsg = payload?.message || null;

        // Clear recording indicator when message sent
        if (recordingMap.has(uid)) {
          const recInfo = recordingMap.get(uid);
          recordingMap.delete(uid);
          if (recInfo?.conversationId) {
            io.to(String(recInfo.conversationId)).emit("voice:recording", { userId: uid, recording: false, conversationId: recInfo.conversationId });
          }
        }

        // âœ… If client already has the DB-inserted message, just broadcast it
        if (convId && providedMsg) {
          io.to(convId).emit("message:new", providedMsg);
          io.to(convId).emit("chat:new_message", { conversationId: convId });
          return;
        }

        // âœ… DM fallback: fetch latest message for the pair (legacy behavior)
        if (convId.startsWith("dm_")) {
          const [_, a, b] = convId.split("_");
          const id1 = Number(a), id2 = Number(b);
          if (!id1 || !id2) return;

          const [rows] = await pool.execute(
            `SELECT id, from_id, to_id, type, body, created_at, read_at, edited, deleted
             FROM messages
             WHERE ((from_id=? AND to_id=?) OR (from_id=? AND to_id=?))
             ORDER BY id DESC
             LIMIT 1`,
            [id1, id2, id2, id1]
          );
          const msg = rows?.[0];
          if (!msg) return;
          io.to(convId).emit("message:new", msg);
          io.to(convId).emit("chat:new_message", { conversationId: convId });
          return;
        }

        // âœ… Group fallback: convId like c:123 => fetch latest by conversation_id
        if (convId.startsWith("c:")) {
          const gid = Number(convId.replace("c:", ""));
          if (!gid) return;

          const [rows] = await pool.execute(
            `SELECT m.id, m.conversation_id, m.from_id, m.to_id, m.type, m.body, m.created_at, m.read_at, m.edited, m.deleted,
                    u.name AS from_name
             FROM messages m
             JOIN users u ON u.id=m.from_id
             WHERE m.conversation_id=?
             ORDER BY m.id DESC
             LIMIT 1`,
            [gid]
          );
          const msg = rows?.[0];
          if (!msg) return;
          io.to(convId).emit("message:new", msg);
          io.to(convId).emit("chat:new_message", { conversationId: convId });
          return;
        }
      } catch (e) {
        console.error("message:send error", e?.message || e);
      }
    });

    /* MESSAGE READ */
    socket.on("message:read", ({ toUserId, conversationId }) => {
      if (toUserId) {
        const targetUid = Number(toUserId);
        // Update DB
        pool.execute(
          "UPDATE messages SET read_at=NOW() WHERE from_id=? AND to_id=? AND read_at IS NULL",
          [targetUid, uid]
        ).catch(()=>{});
        // Emit back to sender so their ticks update instantly (BLUE TICKS)
        emitToUser(targetUid, "message:read", { fromUserId: uid });
      }
      if (conversationId) {
        const cid = String(conversationId).startsWith('c:') ?
          Number(String(conversationId).replace('c:','')) : 0;
        if (cid) {
          pool.execute(
            "INSERT INTO group_message_reads(conversation_id, user_id, last_read_at) VALUES(?,?,NOW()) ON DUPLICATE KEY UPDATE last_read_at=NOW()",
            [cid, uid]
          ).catch(()=>{});
        }
      }
    });

    /* MESSAGE DELETE */
    socket.on("message:delete", ({ messageId, toUserId, conversationId }) => {
      if (!messageId) return;
      if (toUserId) {
        const room = dmRoom(uid, Number(toUserId));
        io.to(room).emit("message:deleted", { messageId: Number(messageId) });
      }
      if (conversationId) {
        io.to(String(conversationId)).emit("message:deleted", { messageId: Number(messageId) });
      }
    });

    /* MESSAGE EDIT */
    socket.on("message:edit", ({ messageId, body, toUserId, conversationId }) => {
      if (!messageId) return;
      if (toUserId) {
        const room = dmRoom(uid, Number(toUserId));
        io.to(room).emit("message:edited", { messageId: Number(messageId), body: String(body||'') });
      }
      if (conversationId) {
        io.to(String(conversationId)).emit("message:edited", { messageId: Number(messageId), body: String(body||'') });
      }
    });

    /* JOIN ROOM */
    socket.on("join_conversation", async ({ conversationId }) => {
      if (!conversationId) return;
      socket.join(String(conversationId));
    });

    /* GROUP: member added */
    socket.on("group:member_added", ({ conversationId, newUserId, groupInfo }) => {
      if (!conversationId || !newUserId) return;
      const room = "c:" + conversationId;
      emitToUser(Number(newUserId), "group:added_to", { conversationId, groupInfo });
      io.to(room).emit("group:member_updated", { conversationId, action: 'added', userId: Number(newUserId) });
    });

    /* GROUP: member left/removed */
    socket.on("group:member_left", ({ conversationId, userId, removedBy }) => {
      if (!conversationId) return;
      const room = "c:" + conversationId;
      const leftUid = Number(userId || uid);
      io.to(room).emit("group:member_updated", {
        conversationId, action: removedBy ? 'removed' : 'left',
        userId: leftUid, removedBy: removedBy || null
      });
      if (leftUid !== uid) {
        emitToUser(leftUid, "group:removed_from", { conversationId });
      }
    });

    /* DISCONNECT */
    socket.on("disconnect", () => {
      const u = socketUser.get(socket.id);
      socketUser.delete(socket.id);
      if (!u) return;
      console.log("User " + u + " disconnected (" + socket.id + ")");

      // Clear voice recording
      if (recordingMap.has(u)) {
        const recInfo = recordingMap.get(u);
        recordingMap.delete(u);
        const clearPayload = { userId: u, recording: false, conversationId: recInfo.roomId };
        io.to(recInfo.roomId).emit("voice:recording", clearPayload);
        const dm3 = recInfo.roomId.match(/^dm_(\d+)_(\d+)$/);
        if (dm3) {
          const [, a, b] = dm3;
          const otherUid = Number(a) === u ? Number(b) : Number(a);
          emitToUser(otherUid, "voice:recording", clearPayload);
        }
      }

      const fullyOffline = setOffline(u, socket.id);
      if (fullyOffline) {
        io.emit("presence:update", { userId: u, online: false });
        pool.execute("UPDATE users SET last_seen=NOW() WHERE id=?", [u]).catch(()=>{});
      }
    });
  });

  server.listen(PORT, () => {
    console.log("Chat server running on port " + PORT);
    console.log("Allowed origins:", ALLOWED_ORIGINS.join(", "));
  });
}

main().catch(err => { console.error(err); process.exit(1); });
